// server.js
const express = require('express');
const cors = require('cors');
const path = require('path');
require('dotenv').config();

// استدعاء قاعدة البيانات والنماذج
const { sequelize } = require('./src/models');

const app = express();

// --- 1. إعدادات Middleware ---
app.use(cors()); // السماح بالاتصال من أي مصدر (للتطوير)
app.use(express.json()); // قراءة البيانات بصيغة JSON
app.use(express.urlencoded({ extended: true })); // قراءة البيانات من النماذج التقليدية

// --- 2. إعداد الملفات الثابتة (Static Files) ---
// هذا السطر مهم جداً لعرض ملفات HTML (مثل login.html, index.html)
app.use(express.static(path.join(__dirname, 'public')));

// هذا السطر مهم جداً لعرض الصور المرفوعة (بدونه لن تظهر الصور)
// الرابط سيكون: http://localhost:3000/uploads/image-name.jpg
app.use('/uploads', express.static(path.join(__dirname, 'public/uploads')));

// --- 3. استدعاء المسارات (Routes) ---
const authRoutes = require('./src/routes/auth.routes');
const reportRoutes = require('./src/routes/report.routes');
const workflowRoutes = require('./src/routes/workflow.routes'); // سننشئه بالأسفل

// تفعيل المسارات
app.use('/api/auth', authRoutes);
app.use('/api/reports', reportRoutes);
app.use('/api/workflow', workflowRoutes); // مسار جديد لإدارة الاعتمادات

// --- 4. نقطة البداية (اختياري) ---
app.get('/', (req, res) => {
    // توجيه المستخدم مباشرة لصفحة تسجيل الدخول
    res.sendFile(path.join(__dirname, 'public', 'login.html'));
});

// --- 5. تشغيل السيرفر ---
const PORT = process.env.PORT || 3000;

app.listen(PORT, async () => {
    console.log(`🚀 الخادم يعمل على: http://localhost:${PORT}`);

    try {
        await sequelize.authenticate();
        console.log('✅ تم الاتصال بقاعدة البيانات MySQL.');

        // ملاحظة: استخدم force: true مرة واحدة فقط إذا أردت حذف الجداول وبناءها من جديد
        // ثم أعدها إلى alter: true
        await sequelize.sync({ alter: true });
        console.log('✅ الجداول متزامنة وجاهزة.');
        
    } catch (error) {
        console.error('❌ خطأ في الاتصال بقاعدة البيانات:', error);
    }
});